﻿/**
 *	@file	utf_ex.d
 *	@brief	std.utf のバグ回避、utf関係の関数追加
 *	@author	tenk@6809.net
 *	@date	2004-03
 *	@note
 *		dmd 0.80現在、toUTF32(wchar[])を使うと無限ループに陥るので
 *		その回避用。
 *
 *		また、std.utfはUTF16のサロゲート・ペア等に未対応のようなので
 *		対応してみたつもりのルーチンを用意。ただ、std.utfのような
 *		十分なエラー文字チェックをしていないので、デフォルトではoff.
 */
module utf_ex;

private:

/// std.utf を使う場合は定義、この場で定義した toUTF を使う場合は未定義.
private version = USE_STD_UTF;


public import std.utf;



/** UTF16->UTF32. wchar文字列をdchar列に変換. 簡易版(ucs2のみを前提)
 * どうも std.utf.toUTF32(wchar[]) はbugってるので、とりあえず代用
 */
public dchar[] toUTF32(wchar[] wbuf)
{
	version (none/*USE_STD_UTF*/) {
		return std.utf.toUTF32(wbufs);
	} else {
		dchar[] dbuf;
		for (int i = 0; i < wbuf.length; i++) {
			dchar d = wbuf[i];
			// どうせ MB文字に変換するから、これらがあっても未対応文字なんだが、お試しで。
			if ((d & 0xFC00) == 0xD800) {
				if (i+1 < wbuf.length && (wbuf[i+1] & 0xFC00) == 0xDC00) {
					i++;
					d = ((d & 0x3FF) << 10) | (wbuf[i] & 0x3ff);
				}
			}
			dbuf ~= d;
		}
		return dbuf;
	}
}


/** UTF8->UTF32 */
public dchar[] toUTF32(char[] buf)
{
	version (USE_STD_UTF) {
		return std.utf.toUTF32(buf);
	} else {
		dchar[] dbuf;
		int er = 0;
		for (int i = 0; i < buf.length;) {
			int c = buf[i++];
			if (c < 0x80) {
				dbuf ~= c;
			} else {
				int getNx() {
					if (i < buf.length) {
						int cc = buf[i++];
						if ((cc & 0xC0) != 0x80) {
							er = i-1;
						}
						return cc & 0x3f;
					}
					return -1;
				}
				if (c < 0xC0) {
					er = i - 1;
					dbuf ~= '?';
					continue;
				}
				int c2 = getNx();
				if (c2 < 0)
					break;
				if (c < 0xE0) {
					c = ((c & 0x1F) << 6) | c2;
					dbuf ~= c;
					continue;
				}
				int c3 = getNx();
				if (c3 < 0)
					break;
				c2 = (c2 << 6) | c3;
				if (c < 0xF0) {
					c = ((c & 0xF) << 12) | c2;
					dbuf ~= c;
					continue;
				}
				int c4 = getNx();
				if (c4 < 0)
					break;
				c2 = (c2 << 6) | c4;
				if (c < 0xF8) {
					c = ((c&7)<<18) | c2;
					dbuf ~= c;
					continue;
				}
				int c5 = getNx();
				if (c5 < 0)
					break;
				c2 = (c2 << 6) | c5;
				if (c < 0xFC) {
					c = ((c&3)<<24) | c2;
					dbuf ~= c;
					continue;
				}
				int c6 = getNx();
				if (c6 < 0)
					break;
				c2 = (c2 << 6) | c6;
				c = ((c&1)<<30) | c2;
				dbuf ~= c;
			}
		}
		if (er > 0) {
			throw new UtfError("invalid UTF-8 value", er);
		}
		return dbuf;
	}
}

/** UTF32->UTF32 */
public dchar[] toUTF32(dchar[] buf)
{
	return buf;
}


/** UTF8->UTF16 */
public wchar[] toUTF16(char[] buf)
{
	version (USE_STD_UTF) {
		return std.utf.toUTF16(buf);
	} else {
		return toUTF16( toUTF32(buf) );
	}
}


/** UTF16->UTF16 */
public wchar[] toUTF16(wchar[] buf)
{
	return buf;
}


/** UTF32->UTF16 */
public wchar[] toUTF16(dchar[] dbuf)
{
	version (USE_STD_UTF) {
		return std.utf.toUTF16(dbuf);
	} else {
		wchar[] wbuf;
		for (int i = 0; i < dbuf.length; i++) {
			dchar c = dbuf[i];
			if ((c >> 16) == 0) {
				wbuf ~= c;
			} else {
				c -= 0x10000;
				wbuf ~= 0xD800 | ((c >> 10) & 0x3FF);
				wbuf ~= 0xDC00 | ( c & 0x3FF);
			}
		}
		return wbuf;
	}
}


/** UTF32->UTF8 */
public char[] toUTF8(dchar[] dbuf)
{
	version (USE_STD_UTF) {
		return std.utf.toUTF8(dbuf);
	} else {
		char[] buf;
		for (int i = 0; i < dbuf.length; i++) {
			dchar c = dbuf[i];
			if (c <= 0x7F) {
				buf ~= c;
			} else if (c <= 0x7FF) {
				buf ~= 0xC0|(c >> 6);
				buf ~= 0x80|(c & 0x3f);
			} else if (c <= 0xFFFF) {
				buf ~= 0xE0|(c>>12);
				buf ~= 0x80|((c>> 6)&0x3f);
				buf ~= 0x80|(c & 0x3f);
			} else if (c <= 0x1fFFFF) {
				buf ~= 0xF0|(c>>18);
				buf ~= 0x80|((c>>12)&0x3f);
				buf ~= 0x80|((c>> 6)&0x3f);
				buf ~= 0x80|(c & 0x3f);
			} else if (c <= 0x3fffFFFF) {
				buf ~= 0xF8|(c >> 24);
				buf ~= 0x80|((c>>18)&0x3f);
				buf ~= 0x80|((c>>12)&0x3f);
				buf ~= 0x80|((c>> 6)&0x3f);
				buf ~= 0x80|(c & 0x3f);
			} else {
				buf ~= 0xFC|(c >> 30);
				buf ~= 0x80|((c>>24)&0x3f);
				buf ~= 0x80|((c>>18)&0x3f);
				buf ~= 0x80|((c>>12)&0x3f);
				buf ~= 0x80|((c>> 6)&0x3f);
				buf ~= 0x80|(c & 0x3f);
			}
		}
		return buf;
	}
}


/** UTF16->UTF8 */
public char[] toUTF8(wchar[] wbuf)
{
	version (USE_STD_UTF) {
		return std.utf.toUTF8(wbuf);
	} else {
		return toUTF8( toUTF32(wbuf) );
	}
}


/** UTF8->UTF8 */
public char[] toUTF8(char[] buf)
{
	return buf;
}


