﻿/**
 *	@file	tcs.d
 *	@brief	TCHAR 用ライブラリ
 *	@author	tenk@6809.net
 *	@date	2004-03
 *	@note
 *		char の代わりに win32.windows.d で定義される TCHAR を
 *		用いるためのライブラリ。TCHARは char/wchar/dchar の何れか。
 *		だが、実質、wchar でないと困難な状態。
 *
 *		機能は
 *		・std.string の(一部)関数の代用品
 *		  toString->toTCS,toStringz->toTCSz, str???->tcs???()
 *		・他国語(日本語)化の為のT("str")による文字列置換.
 *			(tcsInitDicT()で予め辞書を登録)
 *
 *		文字エンコードは Unicode を想定している。
 *		MBC(SJIS)に関しては version=TCHAR_IS_MBCHAR ならば、T("文字列") と
 *		char,wchar,dchar->TCS変換(toTCS) にて MBS 化を行うようにしてるが、
 *		DMD v0.096以降、charのUTF8チェックが強化されたため、ポインタはchar*、
 *		配列は ubyte[] で扱うようにした... 正直やってらんない状態だが。
 *		(ubyteに統一したいが、winヘッダやstd.file.toMBSzがcharなので..)
 *
 *		win-apiを扱う場合は TCHAR=wcharで W系apiにして、libunicows を
 *		リンクするのが無難なんかな...
 *
 *		あと std.string.d の他の関数をどうするか... wstring.d, dstring.dを
 *		用意、あるいはテンプレートな tstring を用意して、それのaliasを用意
 *		するかも... で、これも放置しすぎでやりかけたものが無駄になってるな...
 *
 */

module tcs;

private:

private import std.utf;

// T(char[])で文字列変換を行う処理を入れる場合
version = TCS_CONV_LANG;

// TCHAR=char のときに、UTF8->MBC変換を行うようにする場合は定義.
// version = TCHAR_IS_MBCHAR;

version (TCHAR_IS_MBCHAR) {
	private import mbc;
	public alias char  TCHAR;
	// []配列の場合, UTF8チェックに引っかかるので ubyteで扱うことに
	alias ubyte	TCHARa;
} else {
	version (TCHAR_WIN32) {
		private import win32api;
		//private import win32.ansi.windows;
		//private import win32.windows;
	} else  version (TCHAR_IS_CHAR) {
		public alias char  TCHAR;
	} else version (TCHAR_IS_WCHAR) {
		public alias wchar TCHAR;
	} else version (TCHAR_IS_DCHAR) {
		public alias dchar TCHAR;
	} else {
		public alias wchar TCHAR;
	}
	alias TCHAR TCHARa;
}


/** 文字列定数専用.
 *	場合によっては、言語ごとの文字列置換を行う
 */
public TCHARa[] T(TCHAR[] st)
{
	version (TCS_CONV_LANG) {
		if (tcsDicT_.length > 0) {
			TCHARa[] t = tcsDicT_[st];
			if (t.length > 0) {
				version (TCHAR_IS_MBCHAR) {
					t = toMBS(t);
				}
				return t;
			}
		}
	}
	version (TCHAR_IS_MBCHAR) {
		st = toMBS(st);
	}
	return st;
}



version (none) {
	// template TSTRING を用いる場合...
	private import Tstring;
	public alias TSTRING!(TCHAR) 		tstring;
	public alias tstring.toStringz 		toTz;
	public alias tstring.toString		toT;
	public alias tstring.c_strlen		tcslen;
	public alias tstring.c_strcpy		tcscpy;
	public alias tstring.c_strCfromD	tcsCfromD;

} else {


/** '\0'終端のTCHAR文字列を Dの文字列に変換 */
public TCHARa[] toTCS(TCHAR *s)
{
	return s ? s[0 .. tcslen(s)] : cast(TCHARa[])null;
}


/** char[] -> TCHARa[] */
public TCHARa[] toTCS(char[] s)
{
	if (TCHAR.sizeof == 1) {
		version (TCHAR_IS_MBCHAR) {
			return toMBS(s);
		} else {
			return s;
		}
	} else if (TCHAR.sizeof == 2) {
		return toUTF16(s);
	} else {
		return toUTF32(s);
	}
}


/** wchar[] -> TCHAR[] */
public TCHARa[] toTCS(wchar[] s)
{
	if (TCHAR.sizeof == 1) {
		version (TCHAR_IS_MBCHAR) {
			return toMBS(s);
		} else {
			return toUTF8(s);
		}
	} else if (TCHAR.sizeof == 2) {
		return s;
	} else {
		return toUTF32(s);
	}
}


/** dchar[] -> TCHAR[] */
public TCHARa[] toTCS(dchar[] s)
{
	if (TCHAR.sizeof == 1) {
		version (TCHAR_IS_MBCHAR) {
			return toMBS(s);
		} else {
			return toUTF8(s);
		}
	} else if (TCHAR.sizeof == 2) {
		return toUTF16(s);
	} else {
		return s;
	}
}


/** 整数値を TCHAR文字列化 */
public TCHARa[] toTCS(int val)
{
	int f = 0;
	if (val < 0) {
		f = 1;
		val = -val;
	}
	TCHAR[10] buf;
	uint v = val;
	int  k = 9;
	for (int i = 10; --i >= 0;) {
		int r = v % 10;
		v	  = v / 10;
		buf[i] = r + '0';
		if (r)
			k = i;
	}
	TCHARa[] rslt;
	rslt.length = f + 10 - k;
	TCHARa* p = cast(TCHARa*)rslt;
	rslt[0] = '-';
	rslt[f .. rslt.length] = buf[k .. 10];
	return rslt;
}
unittest {
	TCHARa[] a;
	a = toTCS(0);			assert(a == "0");
	a = toTCS(1);			assert(a == "1");
	a = toTCS(-1);			assert(a == "-1");
	a = toTCS(20);			assert(a == "20");
	a = toTCS(654321);		assert(a == "654321");
	a = toTCS(-654321);		assert(a == "-654321");
	a = toTCS(1234567890);	assert(a == "1234567890");
	a = toTCS(-1234567890);	assert(a == "-1234567890");
}


/** TCHAR文字列 を '\0'終端文字列に変換 */
public TCHAR* toTCSz(TCHARa[] s)
{
	if (s.length == 0)
		return "";
	version (none) {
		// この処理はやっぱり、不味いように思える...
		TCHARa* p = &s[0] + s.length;
		if (*p == 0)
			return s;
	}
	return cast(TCHAR*)(s ~ cast(TCHARa[])"\0");
}
unittest {
	// gc.mallocの挙動に依存で、16バイトのメモリ２つを連続して
	// 取得、文字の終端'\0'が後半のメモリの先頭にある場合の挙動を確認.
	// -release 時のみ(debug時はメモリに管理ヘッダが付くので意味なし)
	if (TCHAR.sizeof == 1) {
		uint[] a;
		char[] b;
		a.length = 4;	/* 16/uint.sizeof */;
		b = "0123456789ABCDE";
		b ~= "F";
		char*  bz = toTCSz(b);
		assert(std.string.strcmp(bz,"0123456789ABCDEF") == 0);
		a[0] = 0x68;	// 'g'
		assert(std.string.strcmp(bz,"0123456789ABCDEF") == 0);
		if (std.string.strcmp(bz,"0123456789ABCDEF") != 0) {
			throw new Error("unittest error: TCHAR* toTCSz(TCHAR[] s)");
		}
	}
}



//-------------------------------

/** '\0'終端のTCHAR文字列のTCHAR数を数える.
 *	※ メモリ確保のためのサイズ取得に用いたりするので、文字数でないので注意.
 */
public uint tcslen(TCHAR* src)
{
	TCHAR *s = src;
	while (*s++) {}
	--s;
	return cast(uint)(s - src);
}



/** '\0'終端のTCHAR文字列のコピー */
public TCHAR* tcscpy(TCHAR* dst, TCHAR* src)
{
	TCHAR* d = dst;
	while (*src)
		*d++ = *src++;
	*d = '\0';
	return dst;
}



/** Dの文字列を cの文字列(eos='\0')スタイルでdstに書き込む。
 *	strcpy(dst, toStringz(src)) という記述する代わり。
 *	(toStringzはちょっとコストが高いかもなんで)
 */
public TCHAR* tcsCfromD(TCHAR* dst, TCHARa[] src, uint dstlen)
{
	TCHAR* d  = dst;
	TCHAR* de = dst + dstlen-1;
	TCHAR* s  = cast(TCHAR*)&src[0];
	TCHAR* se = cast(TCHAR*)&src[0] + src.length;
	while ((d < de) && (s < se) && (*s != '\0'))
		*d++ = *s++;
	*d = '\0';
	return dst;
}


}	// Tstring.d を使うか否かの範囲の終了



//----------------------------------------------------------------------------
// T(char[])で文字列変換を行う処理を入れる場合

version (TCS_CONV_LANG) {

//private int   tcsUseDicFlg_ = 0;		///< 辞書を使う場合、on
private TCHAR[][TCHAR[]] tcsDicT_;		///< 変換辞書


/** T(TCHAR[]) での文字列の変換辞書の登録。
 * @param dic  辞書. 偶数番がキー、奇数が値。
 */
public void tcsInitDicT(TCHAR[][] dic)
{
	tcsDicT_ = null;
	//tcsUseDicFlg_ = 0;
	tcsAddDicT(dic);
}


/** T(TCHAR[]) での文字列の変換辞書へ追加。
 * @param dic  辞書. 偶数番がキー、奇数が値。
 */
public void tcsAddDicT(TCHAR[][] dic)
{
	if (dic.length > 0) {
		for (int i = 0; i < dic.length/2; i++)
			tcsAddDicT(dic[i*2+0], dic[i*2+1]);
	}
}


/** T(TCHAR[]) での文字列の変換辞書へ１項目追加。
 */
public void tcsAddDicT(TCHAR[] key, TCHAR[] val)
{
	//tcsUseDicFlg_ = 1;
	tcsDicT_[key] = val;
}


} // TCS_CONV_LANG
