﻿/**
 *	@file	mbtextfile.d
 *	@brieaf マルチバイト文字対応のテキストファイル・ストリーム
 *	@author tenk@6809.net
 *	@note
 *		・std.stream.Stream を継承したマルチバイト文字テキスト専用版
 *		  現状、open,close,create,readLine系, writeLine系, writeString系, printf のみ対応.
 *		  read,write等低レベルのものは、元のまま呼び出される。
 *	※mbc.d にまとめていたのを分離し、TCHAR 対応に変更したもの。
 *	※ TCHAR が wchar 以外の場合は TCHAR_IS_CHAR のように varsionを設定しておくこと
 *	※ v0.96以降 char のUTF8チェックが強化されたため、内部表現として MBS(SJIS)を使うのは不可。
 */

module mbtextfile;

private:

import std.stream;
import std.string;
//import std.utf;
import utf_ex;
import std.intrinsic;

import mbc;
import tcs;


/// std.stream.Streamのopen,createで渡すファイル名がMBCの必要があれば 定義, UTF8でよければ未定義.
//private version = API_FNAME_MBC;



/// 文字エンコーディングの種類.
public enum MbcType {
	NONE	= 0,		///< 自動チェック、判定失敗、etc
	ASC 	= 1,		///< 0x7fまでしか使っていないテキスト
	UTF8	= 2,
	UTF16LE = 3,
	UTF16BE = 4,
	UTF32LE = 5,
	UTF32BE = 6,

	MBC 	= 7,		///< OSのデフォルトのマルチバイト文字(環境依存)

	UTF 	= UTF8,
	UTF16	= UTF16LE,
	UTF32	= UTF32LE,
	AUTO	= NONE, 	///< 自動チェック指定.

	// お試し.. MbTextFileを継承した先で利用
	MBC_1	= 8,
	MBC_2	= 9,
	MBC_3	= 10,
	MBC_4	= 11,
}



// ---------------------------------------------------------------------------
// ---------------------------------------------------------------------------

/+
class MbTextFileError : StreamError {
	this(char[] msg) { super(msg); }
}
+/


/// Stream の readLine, writeLine,writeString に、文字コード変換を付加したクラス.
public class MbTextFile : std.stream.File {
	this() {
	}

	~this() {
		close();
	}

	this(char[] filename) {
		this(filename, FileMode.In, MbcType.AUTO, MbcType.UTF);
	}

	this(char[] filename, FileMode mode) {
		this(filename, mode, MbcType.AUTO, MbcType.UTF);
	}

	this(char[] filename, FileMode mode, MbcType fileCT) {
		this(filename, mode, fileCT, MbcType.UTF);
	}

	this(char[] filename, FileMode mode, MbcType fileCT, MbcType intrCT) {
		this();
		open(filename, mode, fileCT, intrCT);
	}

	void open(char[] filename, FileMode mode) {
		open(filename, mode, MbcType.AUTO, MbcType.UTF);
	}

	void open(char[] filename, FileMode mode, MbcType fileCT) {
		open(filename, mode, fileCT, MbcType.UTF);
	}

	/** テキストファイル・オープン. */
	void open(
		char[] filename,	 ///< ファイル名
		FileMode mode,		///< In または Out
		MbcType fileCT, 	///< ファイル側の文字エンコーディング種類
		MbcType intrCT) 	///< プログラム側の文字エンコーディング種類
	{
		//debug printf("open %.*s  %d,%d,%d\n", toMBS(filename), mode, fileCT, intrCT);
		debug mbprintf("open %.*s  %d,%d,%d\n", filename, mode, fileCT, intrCT);
		version (API_FNAME_MBC) {
			super.open(toMBS(filename), mode);
		} else {
			//if (TCHAR.sizeof > 1)
				//filename = toUTF8(filename);
			super.open(filename, mode);
		}
		fileMbcType_ = fileCT;
		// 入力するなら、文字コードのチェック.
		if (mode & FileMode.In) {
			checkFileMbcType();
		}
		intrMbcType_ = intrCT;
		//if (fileMbcType_ < MbcType.UTF8)
			//fileMbcType_ = MbcType.UTF8;
	}

	override void close() {
		super.close();
		fileMbcType_  = MbcType.NONE;
		intrMbcType_  = MbcType.NONE;	//MbcType.UTF8;
		bomOfs_ 	  = 0;
		//newFileFlg_	 = 0;
	}

	void create(char[] filename) {
		create(filename, FileMode.Out, MbcType.AUTO, MbcType.UTF);
	}

	void create(char[] filename, FileMode mode) {
		create(filename, mode, MbcType.AUTO, MbcType.UTF);
	}

	void create(char[] filename, FileMode mode, MbcType fileCT) {
		create(filename, mode, fileCT, MbcType.UTF);
	}

	/** ファイルを作成. いまのところ、mode はoutのみ。
	 */
	void create(
		char[] filename,		///< ファイル名
		FileMode mode,			///< (In/)Out
		MbcType fileCT, 		///< ファイル側の文字コード
		MbcType intrCT) 		///< プログラム側の文字コード
	{
		//debug printf("create %.*s  %d,%d,%d\n", toMBS(filename), mode, fileCT, intrCT);
		debug mbprintf("create %.*s  %d,%d,%d\n", filename, mode, fileCT, intrCT);
		if (1) {
			// 読み書き両用モードはなし。書き込みのみとする。
			version (API_FNAME_MBC) {
				super.create(toMBS(filename), mode);
			} else {
				//if (TCHAR.sizeof > 1)
					//filename = toUTF8(filename);
				super.create(filename, mode);
			}
			intrMbcType_ = intrCT;
			fileMbcType_ = fileCT;
			if (fileMbcType_ < MbcType.UTF8 && intrCT >= MbcType.UTF8)
				fileMbcType_ = intrCT;
			if (fileMbcType_ < MbcType.UTF8)
				fileMbcType_ = MbcType.UTF8;

		// ファイル作成.
		} else {
			/+
			// 読み書き両用モード... て、Streamではなしなのかも?
			// ファイルが存在するかチェック.
			if (mode & FileMode.Out) {
				try {
					super.open(filename, mode);
				} catch (OpenError) {
					newFileFlg_ = 1;
				} finally {
					super.close();
				}
			}

			// ファイル作成.
			super.create(filename, mode);

			fileMbcType_ = fileCT;
			intrMbcType_ = intrCT;
			if (newFileFlg_ == 0 && (fileMbcType == MbcType.AUTO && (mode & FileMode.In))) {
				// 読み込み時は文字コードを調べる.
				checkFileMbcType(); // Unicode BOM があった場合はその分ファイルポインタは進めている.
				// 文字コードが決定しない場合は UTF8 とする.
				if (fileMbcType_ < MbcType.UTF8)
					fileMbcType_ = MbcType.UTF8;
			}
			+/
		}

		// 新規オープンとして、Unicodeなら、BOMを込む.
		writeUtfBOM();
		//newFileFlg_ = 0;
	}

	/// 現在のファイル側の文字コード種を返す.
	MbcType fileMbcType() {
		return fileMbcType_;
	}

	// tcs版をデフォルトにするのはやっぱ、やめ
	//alias readLineT readLine;
	alias readLineB readLine;

	TCHAR[] readLineT() {
		if (TCHAR.sizeof == 1)
			return cast(char[])readLineMB();
		else if (TCHAR.sizeof == 2)
			return readLineW();
		else if (TCHAR.sizeof == 4)
			return readLineD();
		else
			assert(0);
	}

	/// UTF8で１行入力.
	char[] readLineB() {
		return cast(char[])readLineMB();
	}

	/// UTF8/MBS のいずれかで１行入力. (基本はMBS)
	ubyte[] readLineMB() {
		ubyte[] buf;
		MbcType ct	= intrMbcType_;
		MbcType fct = fileMbcType_;
		if (fct == MbcType.UTF32LE || fct == MbcType.UTF32BE) {
			dchar[] dbuf = readLineRawD(fct == MbcType.UTF32BE);
			buf = utf32ToCt(dbuf, fct);
		} else if (fct == MbcType.UTF16LE || fct == MbcType.UTF16BE) {
			wchar[] wbuf = readLineRawW(fct == MbcType.UTF16BE);
			buf = utf16ToCt(wbuf, fct);
		} else {
			buf = readLineRawB();
			buf = convCTb(buf, fct, ct);
		}
		return buf;
	}

	/// UTF16で１行入力.
	override wchar[] readLineW() {
		if (fileMbcType_ == MbcType.UTF32LE || fileMbcType_ == MbcType.UTF32BE) {
			return toUTF16( readLineRawD(fileMbcType_ == MbcType.UTF32BE) );
		} else if (fileMbcType_ == MbcType.UTF16LE || fileMbcType_ == MbcType.UTF16BE) {
			return readLineRawW(fileMbcType_ == MbcType.UTF16BE);
		} else {
			return ctToUtf16( readLineRawB(), fileMbcType_ );
		}
	}

	/// UTF32で１行入力.
	dchar[] readLineD() {
		if (fileMbcType_ == MbcType.UTF32LE || fileMbcType_ == MbcType.UTF32BE) {
			return readLineRawD(fileMbcType_ == MbcType.UTF32BE);
		} else if (fileMbcType_ == MbcType.UTF16LE || fileMbcType_ == MbcType.UTF16BE) {
			return toUTF32( readLineRawW(fileMbcType_ == MbcType.UTF16BE) );
		} else {
			return ctToUtf32( readLineRawB(), fileMbcType_ );
		}
	}

	// tcs版をデフォルトにするのはやっぱ、やめ
	//alias writeLineT writeLine;
	alias writeLineB writeLine;

	/// TCHAR で文字列入力
	void writeLineT(TCHAR[] buf) {
		if (TCHAR.sizeof == 1) {
			version (TCHAR_IS_MBCHAR)
				writeLineMB(cast(ubyte[])buf);
			else
				writeLineB(buf);
		} else if (TCHAR.sizeof == 2) {
			writeLineW(buf);
		} else if (TCHAR.sizeof == 4) {
			writeLineD(buf);
		}
	}

	/// MBS/UTF8/UTF16/UTF32 で１行出力
	void writeLineB(char[] buf) {
		writeStringB(buf);
		version (Win32)
			writeStringB("\r\n");
		else version (Mac)
			writeStringB("\r");
		else
			writeStringB("\n");
	}

	/// MBS/UTF8/UTF16/UTF32 で１行出力
	void writeLineMB(ubyte[] buf) {
		writeStringMB(buf);
		version (Win32)
			writeStringB("\r\n");
		else version (Mac)
			writeStringB("\r");
		else
			writeStringB("\n");
	}

	/// MBS/UTF8/UTF16/UTF32 で１行出力
	override void writeLineW(wchar[] wbuf) {
		writeStringW(wbuf);
		version (Win32)
			writeStringW("\r\n");
		else version (Mac)
			writeStringW("\r");
		else
			writeStringW("\n");
	}

	/// MBS/UTF8/UTF16/UTF32 で１行出力
	void writeLineD(dchar[] dbuf) {
		writeStringD(dbuf);
		version (Win32)
			writeStringD("\r\n");
		else version (Mac)
			writeStringD("\r");
		else
			writeStringD("\n");
	}

	// tcs版をデフォルトにするのはやっぱ、やめ
	//alias writeStringT writeString;
	alias writeStringB writeString;

	/// TCHAR文字列を出力
	void writeStringT(TCHAR[] buf) {
		if (TCHAR.sizeof == 1)
			writeStringB(buf);
		else if (TCHAR.sizeof == 2)
			writeStringW(buf);
		else if (TCHAR.sizeof == 4)
			writeStringD(buf);
	}

	/// MBS/UTF8/UTF16/UTF32 で文字列を出力
	void writeStringB(char[] buf0) {
		ubyte[] buf = cast(ubyte[])buf0;
		if (fileMbcType_ == MbcType.UTF32LE || fileMbcType_ == MbcType.UTF32BE) {
			writeStringRawD( ctToUtf32(buf, intrMbcType_), fileMbcType_ == MbcType.UTF32BE);
		} else if (fileMbcType_ == MbcType.UTF16LE || fileMbcType_ == MbcType.UTF16BE) {
			writeStringRawW( ctToUtf16(buf, intrMbcType_), fileMbcType_ == MbcType.UTF16BE);
		} else {
			writeStringRawB( convCTb(buf, intrMbcType_, fileMbcType_) );
		}
	}

	/// MBS をMBS/UTF8/UTF16/UTF32 で文字列を出力
	void writeStringMB(ubyte[] buf) {
		if (fileMbcType_ == MbcType.UTF32LE || fileMbcType_ == MbcType.UTF32BE) {
			writeStringRawD( ctToUtf32(buf, intrMbcType_), fileMbcType_ == MbcType.UTF32BE);
		} else if (fileMbcType_ == MbcType.UTF16LE || fileMbcType_ == MbcType.UTF16BE) {
			writeStringRawW( ctToUtf16(buf, intrMbcType_), fileMbcType_ == MbcType.UTF16BE);
		} else {
			writeStringRawB( convCTb(buf, intrMbcType_, fileMbcType_) );
		}
	}

	/// MBS/UTF8/UTF16/UTF32 で文字列を出力
	override void writeStringW(wchar[] wbuf) {
		if (fileMbcType_ == MbcType.UTF32LE || fileMbcType_ == MbcType.UTF32BE) {
			writeStringRawD( toUTF32(wbuf), fileMbcType_ == MbcType.UTF32BE);
		} else if (fileMbcType_ == MbcType.UTF16LE || fileMbcType_ == MbcType.UTF16BE) {
			writeStringRawW( wbuf, fileMbcType_ == MbcType.UTF16BE);
		} else {
			writeStringRawB( utf16ToCt(wbuf, fileMbcType_) );
		}
	}

	/// MBS/UTF8/UTF16/UTF32 で文字列を出力
	void writeStringD(dchar[] dbuf) {
		if (fileMbcType_ == MbcType.UTF32LE || fileMbcType_ == MbcType.UTF32BE) {
			writeStringRawD( dbuf, fileMbcType_ == MbcType.UTF32BE);
		} else if (fileMbcType_ == MbcType.UTF16LE || fileMbcType_ == MbcType.UTF16BE) {
			writeStringRawW( toUTF16(dbuf), fileMbcType_ == MbcType.UTF16BE);
		} else {
			writeStringRawB( utf32ToCt(dbuf, fileMbcType_) );
		}
	}

	/// １バイト文字の入力.
	ubyte[] readLineRawB() {
		//return super.readLine();
		ubyte[] buf;
		ubyte	bak = 0;
		while (super.eof() == 0) {
			ubyte getC() {
				ubyte[1] buf;
				/*super.*/readBlock(cast(void*)buf, 1);
				return buf[0];
			}
			ubyte c  = getC();	//ubyte c = (bak) ? bak : getC();
								//bak = 0;
			if (c == '\n') {
				break;
			} else if (c == '\r') {
				if (super.eof())
					break;
				bak = getC();
				if (bak == '\n')
					break;
				super.seekCur(-1);
				break;
			}
			buf ~= c;
		}
		return buf;
	}

	/// 2バイト文字(UTF16)の行入力.
	wchar[] readLineRawW(bit bigEn) {
		wchar[] wbuf;
		wchar	bak = 0;
		while (super.eof() == 0) {
			wchar getWc() {
				ubyte[2] buf;
				buf[0] = buf[1] = 0;
				/*super.*/readBlock(cast(void*)buf, 2);
				// dmd v0.79では super. をつけるとコンパイラが落ちる...
				// どうやらメンバー関数中でローカル関数を定義して、
				// その中でsuperを使うとクラッシュするよう.
				// ちなみに this は大丈夫。
				return (bigEn == 0) ? (buf[1]<<8)|buf[0] : (buf[0]<<8)|buf[1];
			}
			wchar wc = (bak) ? bak : getWc();
			bak = 0;

			if (wc == '\n') {
				break;
			} else if (wc == '\r') {
				if (super.eof())
					break;
				bak = getWc();
				if (bak == '\n')
					break;
				super.seekCur(-2);
				break;
			}
			wbuf ~= wc;
		}
		return wbuf;
	}

	/// 4バイト文字(UTF32)の行入力.
	dchar[] readLineRawD(bit bigEn) {
		dchar[] dbuf;
		dchar	dbak = 0;
		while (super.eof() == 0) {
			dchar getDc() {
				ubyte[4] buf;
				buf[0] = buf[1] = buf[2] = buf[3] = 0;
				/*super.*/readBlock(&buf[0], 4);
				return (bigEn == 0) ? (buf[3]<<24)|(buf[2]<<16)|(buf[1]<<8)|buf[0]
								  : (buf[0]<<24)|(buf[1]<<16)|(buf[2]<<8)|buf[3];
			}
			dchar dc = (dbak) ? dbak : getDc();
			dbak = 0;
			if (dc == '\n') {
				break;
			} else if (dc == '\r') {
				if (super.eof())
					break;
				dbak = getDc();
				if (dbak == '\n')
					break;
				super.seekCur(-4);
				break;
			}
			dbuf ~= dc;
		}
		return dbuf;
	}

	/// B文字列書き込み(下位)
	void writeStringRawB(ubyte[] buf) {
		//super.writeString(buf);
		super.writeBlock(&buf[0], buf.length);
	}

	/// W文字列書き込み(下位)
	void writeStringRawW(wchar[] wbuf, bit bigEn) {
		bit swapSw = bigEn;
		version (BigEndian) {
			swapSw ^= 1;
		}
		if (swapSw == 0) {
			super.writeBlock(&wbuf[0], wbuf.length*2);
		} else {
			wchar swaphl(wchar c) {
				return ((c >> 8)&0xFF) | ((c & 0xFF) << 8);
			}
			wchar[] wbuf2;
			wbuf2.length = wbuf.length;
			int i;
			for (i = 0; i < wbuf.length; i++) {
				wchar c = wbuf[i];
				wbuf2[i] = swaphl(c);
			}
			super.writeBlock(&wbuf2[0], wbuf2.length*2);
		}
	}

	/// D文字列書き込み(下位)
	void writeStringRawD(dchar[] dbuf, bit bigEn) {
		bit swapSw = bigEn;
		version (BigEndian) {
			swapSw ^= 1;
		}
		if (swapSw == 0) {
			super.writeBlock(&dbuf[0], dbuf.length*4);
		} else {
			dchar[] dbuf2;
			dbuf2.length = dbuf.length;
			int i;
			for (i = 0; i < dbuf.length; i++) {
				dchar c = dbuf[i];
				dbuf2[i] = bswap(c);
			}
			super.writeBlock(&dbuf2[0], dbuf2.length*4);
		}
	}

private:
	/// 新規オープンで、Unicodeなら、BOMを書き込む.
	void writeUtfBOM() {
		switch (fileMbcType_) {
		case MbcType.UTF8:	  super.writeBlock(cast(ubyte*)x"EFBBBF", 3); break;
		case MbcType.UTF16LE: super.writeBlock(cast(ubyte*)x"FFFE", 2); break;
		case MbcType.UTF16BE: super.writeBlock(cast(ubyte*)x"FEFF", 2); break;
		case MbcType.UTF32LE: super.writeBlock(cast(ubyte*)x"FFFE0000", 4); break;
		case MbcType.UTF32BE: super.writeBlock(cast(ubyte*)x"0000FEFF", 4); break;
		default:	break;
		}
	}

	/// 読み込みオープンで、文字コードを調べる。 とりあえず先頭 32Kバイトから判別.
	void checkFileMbcType() {
		bomOfs_ = 0;
		MbcType ct = fileMbcType_;
		if (1/*ct == MbcType.AUTO*/) {
			// 自動チェックのとき
			ubyte[0x8000] buf;
			uint sz = super.readBlock(&buf[0], buf.length);
			if (sz > 0) {
				ct = checkMbText(buf, sz, bomOfs_);
			}
		} else {
			// タイプ指定されている場合でも、最低限 UTF かどうかはチェックしとく
			ubyte[0x4] buf;
			uint sz = super.readBlock(&buf[0], buf.length);
			if (sz > 0) {
				ct = checkUtfBOM(buf, sz, bomOfs_);
			}
		}
		if (ct < MbcType.UTF8)
			ct = fileMbcType_;
		if (ct < MbcType.UTF8)
			ct = intrMbcType_;
		if (ct < MbcType.UTF8)
			ct = MbcType.UTF8;
		fileMbcType_ = ct;
		super.seekSet(bomOfs_);
	}

	/// 文字コードの種類を指定してchar[]を変換
	ubyte[] convCTb(ubyte[] buf, MbcType sct, MbcType dct) {
		if (dct == MbcType.MBC && sct != MbcType.MBC)
			buf = toMBS(cast(char[])buf);
		else if (dct != MbcType.MBC && sct == MbcType.MBC)
			buf = cast(ubyte[])mbsToUTF8(buf);
		return buf;
	}

	/// UTF16から dct のMB文字コードに変換
	ubyte[] utf16ToCt(wchar[] wbuf, MbcType dct) {
		if (dct == MbcType.MBC)
			return toMBS( wbuf );
		return cast(ubyte[])toUTF8( wbuf );
	}

	/// UTF32から dct のMB文字コードに変換
	ubyte[] utf32ToCt(dchar[] dbuf, MbcType dct)
	{
		if (dct == MbcType.MBC)
			return toMBS( dbuf );
		return cast(ubyte[])toUTF8( dbuf );
	}

	/// MB文字コード(ct)のsをUTF16に変換
	wchar[] ctToUtf16(ubyte[] s, MbcType ct) {
		if (ct == MbcType.MBC)
			return mbsToUTF16(s);
		return toUTF16( cast(char[])s );
	}

	/// MB文字コード(ct)のsをUTF32に変換
	dchar[] ctToUtf32(ubyte[] s, MbcType ct) {
		if (ct == MbcType.MBC)
			return toUTF32(mbsToUTF16(s));
		return toUTF32( cast(char[])s );
	}

private:
	MbcType intrMbcType_;						///< 呼び元に返す文字コード.
	MbcType fileMbcType_;						///< ターゲットファイルの文字コード種.
	int 	bomOfs_ 	= 0;					///< Unicode のBOMによるオフセット.
	//bit	newFileFlg_ = 0;					///< 新規オープンか？.
}




// -------------------------------------------------------------------------
// 日本語テキストの文字コードの種類のチェック.


/** bufをテキストとして、UTFのBOMをチェック */
public MbcType checkUtfBOM(void *mem, int len, inout int bomOfs)
{
	ubyte *buf = cast(ubyte*)mem;
	MbcType ct = MbcType.NONE;

	// Unicode のBOM のチェック.
	if (len >= 2 && buf[0] == 0xff && buf[1] == 0xfe) {
		if (len >= 4 && buf[2] == 0 && buf[3] == 0) {
			ct = MbcType.UTF32LE;
			bomOfs = 4;
		} else {
			ct = MbcType.UTF16LE;
			bomOfs = 2;
		}
	} else if (len >= 2 && buf[0] == 0xfe && buf[1] == 0xff) {
		ct = MbcType.UTF16BE;
		bomOfs = 2;
	} else if (len >= 4 && buf[0] == 0 && buf[1] == 0 && buf[2] == 0xfe && buf[3] == 0xff) {
		ct = MbcType.UTF32BE;
		bomOfs = 4;
	} else if (len >= 3 && buf[0] == 0xEF && buf[1] == 0xBB && buf[2] == 0xBF) {
		ct = MbcType.UTF8;
		bomOfs = 3;
	} else {
		bomOfs = 0;
	}
	return ct;
}


/** bufをテキストとして、コードをチェック */
public MbcType checkMbText(void *mem, int len, inout int bomOfs)
{
	ubyte *buf = cast(ubyte*)mem;
	// まずは UTFのチェック
	MbcType ct = checkUtfBOM(mem, len, bomOfs);
	if (ct > MbcType.NONE)
		return ct;
	len -= bomOfs;
	if (len <= 0)
		return ct;
	buf += bomOfs;

	// UTF8/MBCの判定.
	MbTextCheck_base mbc  = new MbTextCheckMBC(buf, len);
	MbTextCheck_base utf8 = new MbTextCheckUTF8(buf, len);

	if (mbc.level_ > utf8.level_)
		ct = MbcType.MBC;
	else if (mbc.level_ < utf8.level_)
		ct = MbcType.UTF8;
	return ct;
}



/// 文字コードを調べる用
class MbTextCheck_base {
	int  level_ 	= 0;	///< 0:条件を満たさなかった 1:辛うじて満たすが確定できない	2..6:中間  7:絶対的な条件が見つかった.
	//uint basicChrNum_ = 0;	///< 全角のかなや英数記号あたりの数.
	bit  asciiFlg_	= 1;	///< 0x80未満のコードのみならon.
	bit  escFlg_	= 0;	///< 0x1b があったら on.
	bit  warnChrFlg_= 0;	///< 微妙に仕様違反な文字の数.
	bit  binFlg_	= 0;	///< バイナリぽいコードの数.
	bit  spFlg_ 	= 0;	///< 有効な加点情報があった.

protected:
	static ubyte[0x20] ctrlBin_ = [
	  //0 1 2 3  4 5 6 7  8 9 a b  c d e f	0 1 2 3  4 5 6 7  8 9 a b  c d e f
	  //			   a  b t n v  f r
		0,0,0,0, 0,0,0,1, 1,1,1,1, 1,1,0,0, 0,0,0,0, 0,0,0,0, 0,0,1,1, 0,0,0,0,
	];

	// 取得情報からレベルを決定.
	void calcLevel() {
		if (level_ > 0) {
			if (asciiFlg_) {		// 0x7f以下の文字ばかりだった.
				if (binFlg_) {		// バイナリぽい値がある.
					level_ = 1;
				} else if (escFlg_) {	// 0x1bがある.
					level_ = 2;
				} else {			// ascii なんちゃうかなあ.
					level_ = 3;
				}
			} else {
				level_ = 5; 		// 全角判定.
				if (warnChrFlg_ | binFlg_)	// 仕様範囲だが、ちょっと怪しい文字が入っている.
					level_ = 4;
				else if (spFlg_)
					level_ = 6;
			}
		}
	}
};



/// MBCかどうかのチェック.
class MbTextCheckMBC : MbTextCheck_base {
	this(ubyte *top, uint len)
	{
		ubyte *p = top;
		ubyte *btm = top + len - 1;
		int get1() {
			if (p < btm) {
				return *p++;
			}
			return -1;	//終わり.
		}

		level_ = 0;
		for (;;) {
			int c = get1();
			if (c >= 0x80) {
				asciiFlg_ = 0;
				if (ismbblead(c)) {
					int d = get1();
					if (ismbbtrail(d)) {
						c = c << 8 | d;
						//if (d < 0x80)
							//spFlg_ = 1;
						//if (c < 0x8396)
							//basicChrNum_++;
					} else {
						if (d < 0)
							level_ = 1;
						break;
					}
				} else if (c == 0x80 || c == 0xff) {
					binFlg_ = 1;
					break;
				}
			} else {
				if (c >= 0x20) {
					;
				} else if (c < 0) {
					level_ = 1;
					break;
				} else {
					if (ctrlBin_[c]) {
						if (c == 0x1b)
							escFlg_ = 1;
					} else {
						debug .printf("bin %02x\n", c);
						binFlg_ = 1;
					}
				}
			}
		}
		// 終端までチェックした.
		calcLevel();
	}
}


/// utf8かどうかのチェック.
class MbTextCheckUTF8 : MbTextCheck_base {
	this(ubyte *top, uint len)
	{
		ubyte *p = top;
		ubyte *btm = top + len - 1;
		int get1() {
			if (p < btm) {
				return *p++;
			}
			return -1;	//終わり.
		}
		int getNx() {
			if (p < btm) {
				int c = *p++;
				if ((c & 0xC0) == 0x80)
					return c & 0x3f;
				return 0;	// エラーだった.
			}
			return -1;	//終わり.
		}

		level_ = 0;
		for (;;) {
			int c = get1();
			if (c >= 0x80) {
				asciiFlg_ = 0;
				if (c < 0xC0) {
					break;
				}
				int c2 = getNx();
				if (c2 <= 0) {
					if (c2 < 0)
						level_ = 1;
					break;
				}
				if (c < 0xE0) {
					c = ((c & 0x1F) << 6) | c2;
					if (c < 0x7F)
						warnChrFlg_ = 1;
				} else {
					int c3 = getNx();
					if (c3 <= 0) {
						if (c3 < 0)
							level_ = 1;
						break;
					}
					if (c < 0xF0) {
						c = ((c & 0xF) << 12) | (c2 << 6) | c3;
						if (c < 0x7FF)
							warnChrFlg_ = 1;
						//if (c >= 0xff61 && c <= 0xff9f)
							//hankanaNum_++;
					} else {
						int c4 = getNx();
						if (c4 <= 0) {
							if (c4 < 0)
								level_ = 1;
							break;
						}
						if (c < 0xF8) {
							c = ((c&7)<<18) | (c2<<12) | (c3<<6) | c4;
							if (c <= 0xFFFF)
								warnChrFlg_ = 1;
						} else {
							int c5 = getNx();
							if (c5 <= 0) {
								if (c5 < 0)
									level_ = 1;
								break;
							}
							if (c < 0xFC) {
								c = ((c&3)<<24) | (c2<<18) | (c3<<12) | (c4<<6) | c5;
								if (c <= 0x1fFFFF)
									warnChrFlg_ = 1;
							} else {
								int c6 = getNx();
								if (c6 <= 0) {
									if (c6 < 0)
										level_ = 1;
									break;
								}
								c = ((c&1)<<30) |(c2<<24) | (c3<<18) | (c4<<12) | (c5<<6) | c6;
								if (c <= 0x3fffFFFF) {
									warnChrFlg_ = 1;
								}
							}
						}
					}
				}
			} else {
				if (c >= 0x20) {
					;
				} else if (c < 0) {
					level_ = 1;
					break;
				} else {
					if (ctrlBin_[c]) {
						if (c == 0x1b)
							escFlg_ = 1;
					} else {
						binFlg_ = 1;
					}
				}
			}
		}

		//if (warnChrFlg_ == 0) // UTF8を優先する.
			//spFlg_ = 1;
		calcLevel();
	}
}



