﻿/**
 *	@file	mbc.d
 *	@brieaf とりあえずのマルチバイト文字対策のサブルーチンズ.
 *	@author tenk@6809.net
 *	@date	2004/02/08
 *	@note
 *		・mbs<->utf文字列の文字コード変換
 *		・(ほぼ) Win専用.	linux の setlocale や wcstombs,mbstowcsの都合がわからないので...
 */

module mbc;

private:
private import std.string;
private import std.utf;


// ---------------------------------------------------------------------------
// ---------------------------------------------------------------------------
// OS環境依存のマルチバイト文字とUTFの変換。
//	http://www.sun-inet.or.jp/~yaneurao/dlang/
//	「 文字コードにまつわるエトセトラ」のを改造...ゆえ、locale_sync_objectを
//	 上記のソースと一緒に使う場合は要修正になる(locale_sync_object)。


version (Win32) {
	import std.c.windows.windows;
	//import win32api;
	extern (C) {
		int _ismbblead(char b);
		int _ismbbtrail(char b);
	}
} else {
	// ノーチェック
	extern (C) {
		enum { LC_ALL = 0 }
		char* setlocale(int, char*);
		int mbstowcs(wchar*, char*, int);
		int wcstombs(char*, wchar*, int);
		// ? あるのか？
		int _ismbblead(char b);
		int _ismbbtrail(char b);
	}
	static Object locale_sync_object;
}


// char が UTF8 かどうかチェックするようになってしまったので、面倒だが mbcは ubyte で扱う...
//public alias char	MBChar;
public alias ubyte	MBChar;


/// MBC全角の上位一バイトに適合するか?
public int ismbblead(char b) {
	return _ismbblead(b);
}

//	/// MBC全角の上位一バイトに適合するか?
//	public int ismbblead(MBChar b) {
//		return _ismbblead(b);
//	}


/// MBC全角の下位一バイトに適合するか?
public int ismbbtrail(char b) {
	return _ismbbtrail(b);
}

//	/// MBC全角の下位一バイトに適合するか?
//	public int ismbbtrail(MBChar b) {
//		return _ismbbtrail(b);
//	}



/// MBC->UTF8
public char[] mbsToUTF8(char* buf)
{
	return mbsToUTF8(cast(ubyte*)buf);
}

/// MBC->UTF8
public char[] mbsToUTF8(MBChar* buf)
{
	if (buf)
		return mbsToUTF8(buf[0 .. std.string.strlen(cast(char*)buf)]);
	return "";
}

/// MBC->UTF8
public char[] mbsToUTF8(char[] buf)
{
	return mbsToUTF8(cast(MBChar[])buf);
}

/// MBC->UTF8
public char[] mbsToUTF8(MBChar[] buf)
{
	return toUTF8( mbsToUTF16(buf) );
}


/// MBC->UTF16
public wchar[] mbsToUTF16(MBChar* buf)
{
	if (buf)
		return mbsToUTF16(buf[0 .. std.string.strlen(cast(char*)buf)]);
	return "";
}

/// MBC(multi byte char.) string -> UTF16 string
public wchar[] mbsToUTF16(char[] buf)
{
	return mbsToUTF16(cast(MBChar[])buf);
}

/// MBC(multi byte char.) string -> UTF16 string
public wchar[] mbsToUTF16(MBChar[] buf)
{
	wchar[] wbuf;
	version (Win32) {
		const int cp = 0;
		wbuf.length = MultiByteToWideChar(cp, 0, cast(char*)buf, buf.length, null, 0);
		MultiByteToWideChar(cp, 0, cast(char*)buf, buf.length, cast(wchar*)wbuf, wbuf.length);
	} else {
		// ノーチェック
		synchronized (locale_sync_object) {
			char* o = setlocale(LC_ALL, null);
			setlocale(LC_ALL, "");
			wbuf.length = mbstowcs(null, buf, 0);
			mbstowcs(wbuf, buf, wbuf.length);
			setlocale(LC_ALL, o);
		}
	}
	return wbuf;
}


/// MBC->UTF8
public dchar[] mbsToUTF32(MBChar* buf)
{
	if (buf)
		return mbsToUTF32(buf[0 .. std.string.strlen(cast(char*)buf)]);
	return "";
}

/// MBC->UTF32
public dchar[] mbsToUTF32(char[] buf)
{
	return mbsToUTF32(cast(MBChar[])buf);
}

/// MBC->UTF32
public dchar[] mbsToUTF32(MBChar[] buf)
{
	return toUTF32( mbsToUTF16(buf) );
}



/// UTF16->MBC String
public MBChar[] toMBS(wchar[] wbuf)
{
	MBChar[] buf;
	version (Win32) {
		const uint cp = 0;
		buf.length = WideCharToMultiByte(cp, 0, cast(wchar*)wbuf, wbuf.length, null, 0, null, null);
		WideCharToMultiByte(cp, 0, cast(wchar*)wbuf, wbuf.length, cast(char*)buf, buf.length, null, null);
	} else {
		// ノーチェック
		synchronized (locale_sync_object) {
			char* o = setlocale(LC_ALL, null);
			setlocale(LC_ALL, "");
			buf.length = wcstombs(null, wbuf, 0);
			wcstombs(buf, wbuf, buf.length);
			setlocale(LC_ALL, o);
		}
	}
	return buf;
}


/// UTF8->MBC String
public MBChar[] toMBS(char[] buf)
{
	return toMBS( toUTF16(buf) );
}


/// UTF32->MBC String
public MBChar[] toMBS(dchar[] dbuf)
{
	return toMBS( toUTF16(dbuf) );
}


// toMBSz も MBChar* を返すほうが自然だろうが、std.file の同名関数が char*を
// 返すので一応、合わせておく...

/// UTF16->MBC String(eos='\0')
public char* toMBSz(wchar* s)
{
	//if (s) return std.string.toStringz(toMBS(s[0 .. std.string.wcslen(s)]));
	if (s) {
		return cast(char*)toMBS(s[0 .. std.string.wcslen(s)+1]);
	}
	return cast(char*)"";
}

/// UTF16->MBC String(eos='\0')
public char *toMBSz(wchar[] wbuf)
{
	//return	std.string.toStringz( toMBS(wbuf) );
	wchar[] wb = wbuf.dup;
	wb ~= '\0';
	return cast(char*)toMBS( wb );
}

/// UTF8->MBC String(eos='\0')
public char * toMBSz(char* s)
{
	//if (s) return std.string.toStringz(toMBS(s[0 .. std.string.strlen(s)]));
	if (s) {
		return cast(char*)toMBS(s[0 .. std.string.strlen(s)+1]);
	}
	return cast(char*)"";
}

/// UTF8->MBC String(eos='\0')
public char* toMBSz(char[] buf)
{
	return toMBSz( toUTF16(buf) );
}

/// UTF32->MBC String(eos='\0')
public char* toMBSz(dchar[] dbuf)
{
	return toMBSz( toUTF16(dbuf) );
}




// ---------------------------------------------------------------------------
// std.file.printf を参考に MBC用 printf を用意

private import std.c.stdlib;
private import std.c.stdio;
private alias std.c.stdio.va_list va_list;


/** UTF8で指定された文字列を MBC にして出力する printf */
public uint mbprintf(char[] fmt, ...)
{
	va_list ap;
	ap = cast(va_list) &fmt;
	ap += fmt.sizeof;
	MBChar[] buf = toMBS( vsprintf(fmt, ap) );
	printf("%.*s", buf);
	return buf.length;
}



/+
char[] sprintf(char[] fmt, ...)
{
	va_list ap;
	ap = cast(va_list) &fmt;
	ap += fmt.sizeof;
	return vsprintf(fmt, ap);
}
+/

char[] vsprintf(char[] fmt0, va_list args)
{
	char[1024]	buf;
	char*		fmt = toStringz(fmt0);
	char*		p	= buf;
	uint		psize = buf.length;
	int 		count;
	for (;;) {
		version (Win32) {
			count = _vsnprintf(p, psize, fmt, args);
		} else version (linux) {
			count = vsnprintf(p, psize, fmt, args);
		} else {
			static assert(0);
		}
		if (count < 0)
			psize *= 2;
		else if (count >= psize)
			psize = count + 1;
		else
			break;
		p = cast(char*) alloca(psize);
	}
	return p[0 .. count];
}


